#include "ddhundoabledivloop.h"
#include "dhlistsorted.h"
#include "dhobjectcopy.h"
#include <stdlib.h>
#include <stdio.h>


//---------------------------------------------------------------------------------------
DDHUndoableDivLoop::DDHUndoableDivLoop()
{
   mSelectMode = ac_get_select_mode();
   mUndoCalled = false;
   mObjsOld = NULL;
   mObjsNew = NULL;
   mSelOld = NULL;
   mSelNew = NULL;
}
//---------------------------------------------------------------------------------------
DDHUndoableDivLoop::~DDHUndoableDivLoop()
{
   for( List *po = mObjsNew; po != NULL; po = po->next )
   {
      t_mObjectParent *op = ( t_mObjectParent * )( po->data );
      if( mUndoCalled )
      {
         object_free( op->o );
      }
      delete op;
   }
   for( List *po = mObjsOld; po != NULL; po = po->next )
   {
      t_mObjectParent *op = ( t_mObjectParent * )( po->data );
      if( !mUndoCalled )
      {
         object_free( op->o );
      }
      delete op;
   }
   list_free( &mObjsOld );
   list_free( &mObjsNew );
   list_free( &mSelOld );
   list_free( &mSelNew );
}
//---------------------------------------------------------------------------------------
void DDHUndoableDivLoop::execute()
{

   // Grab all whole surfaces
   List *surflist = ac_selection_get_whole_surfaces_all();

   // Bail if nothing selected
   if( surflist == NULL )
   {
      message_dialog( "divideloop : no surfaces selected" );
      abort();
      return;
   }

   //////////////////////////////////////////////////////////////////////////
   // OBJECT COPY
   // Copy all of the objects that will be affected by this operation for Undo/Redo.
   // In copying the objects, we also have to copy the selection, so the operation isn't
   // as trivial as object_clone().
   // However, this Object-level-undo method seems superior to performing Undo's on edge inserts and
   // surface removal creation, which gets mildly ugly with edge inserts and
   // unselected surfaces being affected (especially re-inserting the verts on Redo).
   //////////////////////////////////////////////////////////////////////////

   //////////////////////////////////////////////////////////////////////////
   // Grab a list of objects that are partially selected

   // Get a list of all objects for fast-searching
   List *objlist = NULL;
   List *worldlist = NULL;
   list_add_item( &worldlist, ac_get_world() );
   dh_get_objects_from_groups( &objlist, worldlist );
   list_free( &worldlist );
   DHListSorted *s_objlist = new DHListSorted( objlist );
   list_free( &objlist );

   // Collect a list of all objects that are partially selected
   for( List *ps = surflist; ps != NULL; ps = ps->next )
   {
      Surface *s = ( Surface * )( ps->data );
      int idx = s_objlist->mSearch( object_of_surface( s ));
      if( idx > -1 && s_objlist->mTag[idx] == 0 )
      {
         t_mObjectParent *op = new t_mObjectParent;
         op->o = ( ACObject * )( s_objlist->mData[idx] );
         op->parent = object_parent( op->o );
         list_add_item_head( &mObjsOld, op );
         // Mark the object as selected
         s_objlist->mTag[idx] = 1;
      }
   }

   //////////////////////////////////////////////////////////////////////////
   // Populate the old selection list
   switch( mSelectMode )
   {
   case SELECT_SURFACE:
      mSelOld = ac_selection_get_surfaces();
      break;
   case SELECT_VERTEX:
      mSelOld = ac_selection_get_vertices();
      break;
   case SELECT_GROUP:
      mSelOld = ac_selection_get_groups();
      break;
   default:
      mSelOld = ac_selection_get_objects();
      break;
   }

   //////////////////////////////////////////////////////////////////////////
   // Remove the old objects from the document, add the new objects
   List *objclones = NULL;
   for( List *pop = mObjsOld; pop != NULL; pop = pop->next )
   {
      t_mObjectParent *op = ( t_mObjectParent * )( pop->data );
      DHObjectCopy *oc = new DHObjectCopy( op->o, true );
      list_add_item_head( &objclones, oc );
      t_mObjectParent *nop = new t_mObjectParent;
      nop->o = oc->mObject;
      nop->parent = op->parent;
      list_add_item_head( &mObjsNew, nop );
      object_add_child( op->parent, oc->mObject );
      object_remove_child_nocleanup( op->parent, op->o );
   }

   //////////////////////////////////////////////////////////////////////////
   // Duplicate the old selection onto our new objects
   clear_selection();
   List *sellist = NULL;
   switch( mSelectMode )
   {
   case SELECT_SURFACE:
      for( List *poc = objclones; poc != NULL; poc = poc->next )
      {
         DHObjectCopy *oc = ( DHObjectCopy * )( poc->data );
         oc->AppendSelectedSurfaces( &sellist );
      }
      ac_selection_select_surfacelist( sellist );
      break;
   case SELECT_VERTEX:
      for( List *poc = objclones; poc != NULL; poc = poc->next )
      {
         DHObjectCopy *oc = ( DHObjectCopy * )( poc->data );
         oc->AppendSelectedVertices( &sellist );
      }
      ac_selection_select_vertexlist( sellist );
      break;
   case SELECT_GROUP:
      for( List *poc = objclones; poc != NULL; poc = poc->next )
      {
         DHObjectCopy *oc = ( DHObjectCopy * )( poc->data );
         list_add_item_head( &sellist, object_top_ancestor( oc->mObject ));
      }
      ac_selection_select_objectlist( sellist );
      break;
   default:
      for( List *poc = objclones; poc != NULL; poc = poc->next )
      {
         DHObjectCopy *oc = ( DHObjectCopy * )( poc->data );
         list_add_item_head( &sellist, oc->mObject );
      }
      ac_selection_select_objectlist( sellist );
      break;
   }
   list_free( &sellist );

   //////////////////////////////////////////////////////////////////////////
   // Remove the object clone list
   for( List *poc = objclones; poc != NULL; poc = poc->next )
   {
      DHObjectCopy *oc = ( DHObjectCopy * )( poc->data );
      delete oc;
   }
   list_free( &objclones );

   //////////////////////////////////////////////////////////////////////////
   // Cleanup
   list_free( &surflist );

   //////////////////////////////////////////////////////////////////////////
   // LOOP DIVISION
   // Undo/redo work is done; now onto the main function.
   //////////////////////////////////////////////////////////////////////////

   // Grab all whole surfaces
   surflist = ac_selection_get_whole_surfaces_all();

   // Bail if nothing selected
   if( surflist == NULL )
   {
      message_dialog( "divideloop : an internal error occurred (cannot re-establish new selection).  Please Undo the operation and contact the author for support." );
      return;
   }

   //////////////////////////////////////////////////////////////////////////
   // Push surflist into a sorted list
   DHListSorted s_surflist( surflist );

   // Get the selected edges
   List *edge_master = ac_selection_get_edges( FALSE );

   // Get list of selected edges that are connected to at least 2 surfaces,
   // at least one of which is 3- or 4-sided
   List *edgelist = NULL;
   for( List *pe = edge_master; pe != NULL; pe = pe->next )
   {
      ACEdge *e = ( ACEdge * )( pe->data );
      if( e->connected > 1 )
      {
         int num_sel_conn = 0;
         int num_triquad = 0;
         for( List *ps = e->surfacelist; ps != NULL; ps = ps->next )
         {
            // Ensure at least 2 surfaces are connected to this edge
            Surface *s = ( Surface * )( ps->data );
            if( s_surflist.mSearch( s ) > -1 )
            {
               ++num_sel_conn;
               if( s->numvert > 2 && s->numvert < 5 )
               {
                  ++num_triquad;
               }
               if( num_sel_conn > 1 && num_triquad > 0 )
               {
                  list_add_item_head( &edgelist, e );
                  break;
               }
            }
         }
      }
   }

   // Create a list sorted by surface containing all edges per-surface (for fast searching)
   DHSurfaceEdgesList selist( surflist, s_surflist.mCount );
   selist.Sort();

   // Add our edges to surface/edge list
   for( List *pe = edgelist; pe != NULL; pe = pe->next )
   {
      selist.AddEdge(( ACEdge * )( pe->data ));
   }

   // Push identified edges into sorted list
   DHListSorted *s_edgelist = new DHListSorted( edgelist );

   // Add edges that are part of quads with an edge opposite a selected edge
   for( List *pe = edge_master; pe != NULL; pe = pe->next )
   {
      if( s_edgelist->mSearch( pe->data ) > -1 )
      {
         // Edge already marked - ignore
         continue;
      }
      ACEdge *e = ( ACEdge * )( pe->data );
      bool foundedge = false;
      for( List *ps = e->surfacelist; ps != NULL; ps = ps->next )
      {
         Surface *s = ( Surface * )( ps->data );
         if( s->numvert != 4 )
         {
            continue;
         }
         int idx = selist.BSearch( s );
         if( idx > -1 )
         {
            DHSurfaceEdges *se = &( selist.mData[idx] );
            for( List *pe2 = se->edges; pe2 != NULL; pe2 = pe2->next )
            {
               ACEdge *e2 = ( ACEdge * )( pe2->data );
               if( e->v1 != e2->v1 && e->v1 != e2->v2 &&
                   e->v2 != e2->v1 && e->v2 != e2->v2 )
               {
                  // Edge is on opposite side of a selected edge - add and continue
                  foundedge = true;
                  break;
               }
            }
            if( foundedge )
            {
               break;
            }
         }
      }
      if( foundedge )
      {
         selist.AddEdge( e );
         list_add_item_head( &edgelist, e );
      }
   }

   // We've now collected all edges between which vertices should be inserted.
   // Resort the list into a new sorted list instance
   delete s_edgelist;
   s_edgelist = new DHListSorted( edgelist );
   list_free( &edgelist );

   // Create a sorted list of surfaces with 5 or more sides.
   // Use this list to reject 5+ sided surfaces later
   List *reject_surfs = NULL;
   for( int ix = 0; ix < s_surflist.mCount; ++ix )
   {
      Surface *s = ( Surface * )( s_surflist.mData[ix] );
      if( s->numvert > 4 )
      {
         list_add_item_head( &reject_surfs, s );
      }
   }
   DHListSorted s_reject_surfs( reject_surfs );
   list_free( &reject_surfs );

   // Insert vertices in edgelist
   Boolean bnew = FALSE;
   List *new_verts = NULL;
   for( int ix = 0; ix < s_edgelist->mCount; ++ix )
   {
      ACEdge *e = ( ACEdge * )( s_edgelist->mData[ix] );
      Vertex *v = ac_edge_insert_vertex( e, &bnew );
      //list_add_item_head( &mVertsNew, dlv );
      list_add_item_head( &new_verts, v );
   }
   delete s_edgelist;

   // Sort the vertex list
   DHListSorted s_vertlist( new_verts );

   // Divide the surfaces
   int newvertpos[4];
   SVertex *newsurfverts[8];
   SVertex *tmpsurfverts[8];
   List *new_surfs = NULL;
   for( List *ps = surflist; ps != NULL; ps = ps->next )
   {
      Surface *s = ( Surface * )( ps->data );
      if( s->numvert < 4 || s_reject_surfs.mSearch( s ) > -1 )
      {
         // If we added any verts, this number will be > 3
         // Or, if the surface is in the reject list, skip
         continue;
      }
      // Get number of new vertices and their positions in the surface
      int newverts = 0;
      int idx;
      int vidx = 0;
      for( List *psv = s->vertlist; psv != NULL; psv = psv->next )
      {
         SVertex *sv = ( SVertex * )( psv->data );
         idx = s_vertlist.mSearch( sv->v );
         if( idx > -1 )
         {
            newvertpos[newverts] = vidx;
            ++newverts;
         }
         tmpsurfverts[vidx] = sv;
         ++vidx;
      }
      // Arrange the verts so the new one is at pos 0.
      for( int ix = 0; ix < s->numvert; ++ix )
      {
         newsurfverts[ix] = tmpsurfverts[( ix + newvertpos[0] ) % s->numvert];
      }
      if( newverts > 0 )
      {
         ACObject *dls_parent = object_of_surface( s );
         Surface *dls_surf;
         // Add the vertices to the surfaces
         switch( newverts )
         {
         case 1:
            {
               // A triangle with one new vertex was added
               // Split into 2 tris
               dls_surf = new_surface();
               list_add_item_head( &new_surfs, dls_surf );
               surface_add_vertex( dls_surf, newsurfverts[0]->v, newsurfverts[0]->tx, newsurfverts[0]->ty );
               surface_add_vertex( dls_surf, newsurfverts[1]->v, newsurfverts[1]->tx, newsurfverts[1]->ty );
               surface_add_vertex( dls_surf, newsurfverts[2]->v, newsurfverts[2]->tx, newsurfverts[2]->ty );
               surface_copy_properties( s, dls_surf );
               object_add_surface_head( dls_parent, dls_surf );
               dls_surf = new_surface();
               list_add_item_head( &new_surfs, dls_surf );
               surface_add_vertex( dls_surf, newsurfverts[0]->v, newsurfverts[0]->tx, newsurfverts[0]->ty );
               surface_add_vertex( dls_surf, newsurfverts[2]->v, newsurfverts[2]->tx, newsurfverts[2]->ty );
               surface_add_vertex( dls_surf, newsurfverts[3]->v, newsurfverts[3]->tx, newsurfverts[3]->ty );
               surface_copy_properties( s, dls_surf );
               object_add_surface_head( dls_parent, dls_surf );
            }
            break;
         case 2:
            {
               // Either a triangle with two new vertices was added
               if( s->numvert == 5 )
               {
                  // Make it look best aesthetically by making only one division
                  if( newvertpos[1] - newvertpos[0] == 2 )
                  {
                     dls_surf = new_surface();
                     list_add_item_head( &new_surfs, dls_surf );
                     surface_add_vertex( dls_surf, newsurfverts[0]->v, newsurfverts[0]->tx, newsurfverts[0]->ty );
                     surface_add_vertex( dls_surf, newsurfverts[1]->v, newsurfverts[1]->tx, newsurfverts[1]->ty );
                     surface_add_vertex( dls_surf, newsurfverts[2]->v, newsurfverts[2]->tx, newsurfverts[2]->ty );
                     surface_copy_properties( s, dls_surf );
                     object_add_surface_head( dls_parent, dls_surf );
                     dls_surf = new_surface();
                     list_add_item_head( &new_surfs, dls_surf );
                     surface_add_vertex( dls_surf, newsurfverts[0]->v, newsurfverts[0]->tx, newsurfverts[0]->ty );
                     surface_add_vertex( dls_surf, newsurfverts[2]->v, newsurfverts[2]->tx, newsurfverts[2]->ty );
                     surface_add_vertex( dls_surf, newsurfverts[3]->v, newsurfverts[3]->tx, newsurfverts[3]->ty );
                     surface_add_vertex( dls_surf, newsurfverts[4]->v, newsurfverts[4]->tx, newsurfverts[4]->ty );
                     surface_copy_properties( s, dls_surf );
                     object_add_surface_head( dls_parent, dls_surf );
                  }
                  else
                  {
                     dls_surf = new_surface();
                     list_add_item_head( &new_surfs, dls_surf );
                     surface_add_vertex( dls_surf, newsurfverts[0]->v, newsurfverts[0]->tx, newsurfverts[0]->ty );
                     surface_add_vertex( dls_surf, newsurfverts[1]->v, newsurfverts[1]->tx, newsurfverts[1]->ty );
                     surface_add_vertex( dls_surf, newsurfverts[2]->v, newsurfverts[2]->tx, newsurfverts[2]->ty );
                     surface_add_vertex( dls_surf, newsurfverts[3]->v, newsurfverts[3]->tx, newsurfverts[3]->ty );
                     surface_copy_properties( s, dls_surf );
                     object_add_surface_head( dls_parent, dls_surf );
                     dls_surf = new_surface();
                     list_add_item_head( &new_surfs, dls_surf );
                     surface_add_vertex( dls_surf, newsurfverts[0]->v, newsurfverts[0]->tx, newsurfverts[0]->ty );
                     surface_add_vertex( dls_surf, newsurfverts[3]->v, newsurfverts[3]->tx, newsurfverts[3]->ty );
                     surface_add_vertex( dls_surf, newsurfverts[4]->v, newsurfverts[4]->tx, newsurfverts[4]->ty );
                     surface_copy_properties( s, dls_surf );
                     object_add_surface_head( dls_parent, dls_surf );
                  }
               }
               // Or a quad with 2 new vertices was added
               else
               {
                  dls_surf = new_surface();
                  list_add_item_head( &new_surfs, dls_surf );
                  surface_add_vertex( dls_surf, newsurfverts[0]->v, newsurfverts[0]->tx, newsurfverts[0]->ty );
                  surface_add_vertex( dls_surf, newsurfverts[1]->v, newsurfverts[1]->tx, newsurfverts[1]->ty );
                  surface_add_vertex( dls_surf, newsurfverts[2]->v, newsurfverts[2]->tx, newsurfverts[2]->ty );
                  surface_add_vertex( dls_surf, newsurfverts[3]->v, newsurfverts[3]->tx, newsurfverts[3]->ty );
                  surface_copy_properties( s, dls_surf );
                  object_add_surface_head( dls_parent, dls_surf );
                  dls_surf = new_surface();
                  list_add_item_head( &new_surfs, dls_surf );
                  surface_add_vertex( dls_surf, newsurfverts[0]->v, newsurfverts[0]->tx, newsurfverts[0]->ty );
                  surface_add_vertex( dls_surf, newsurfverts[3]->v, newsurfverts[3]->tx, newsurfverts[3]->ty );
                  surface_add_vertex( dls_surf, newsurfverts[4]->v, newsurfverts[4]->tx, newsurfverts[4]->ty );
                  surface_add_vertex( dls_surf, newsurfverts[5]->v, newsurfverts[5]->tx, newsurfverts[5]->ty );
                  surface_copy_properties( s, dls_surf );
                  object_add_surface_head( dls_parent, dls_surf );
               }
            }
            break;
         case 3:
            {
               // A triangle with three new vertices was added
               dls_surf = new_surface();
               list_add_item_head( &new_surfs, dls_surf );
               surface_add_vertex( dls_surf, newsurfverts[0]->v, newsurfverts[0]->tx, newsurfverts[0]->ty );
               surface_add_vertex( dls_surf, newsurfverts[1]->v, newsurfverts[1]->tx, newsurfverts[1]->ty );
               surface_add_vertex( dls_surf, newsurfverts[2]->v, newsurfverts[2]->tx, newsurfverts[2]->ty );
               surface_copy_properties( s, dls_surf );
               object_add_surface_head( dls_parent, dls_surf );
               dls_surf = new_surface();
               list_add_item_head( &new_surfs, dls_surf );
               surface_add_vertex( dls_surf, newsurfverts[2]->v, newsurfverts[2]->tx, newsurfverts[2]->ty );
               surface_add_vertex( dls_surf, newsurfverts[3]->v, newsurfverts[3]->tx, newsurfverts[3]->ty );
               surface_add_vertex( dls_surf, newsurfverts[4]->v, newsurfverts[4]->tx, newsurfverts[4]->ty );
               surface_copy_properties( s, dls_surf );
               object_add_surface_head( dls_parent, dls_surf );
               dls_surf = new_surface();
               list_add_item_head( &new_surfs, dls_surf );
               surface_add_vertex( dls_surf, newsurfverts[4]->v, newsurfverts[4]->tx, newsurfverts[4]->ty );
               surface_add_vertex( dls_surf, newsurfverts[5]->v, newsurfverts[5]->tx, newsurfverts[5]->ty );
               surface_add_vertex( dls_surf, newsurfverts[0]->v, newsurfverts[0]->tx, newsurfverts[0]->ty );
               surface_copy_properties( s, dls_surf );
               object_add_surface_head( dls_parent, dls_surf );
               dls_surf = new_surface();
               list_add_item_head( &new_surfs, dls_surf );
               surface_add_vertex( dls_surf, newsurfverts[0]->v, newsurfverts[0]->tx, newsurfverts[0]->ty );
               surface_add_vertex( dls_surf, newsurfverts[2]->v, newsurfverts[2]->tx, newsurfverts[2]->ty );
               surface_add_vertex( dls_surf, newsurfverts[4]->v, newsurfverts[4]->tx, newsurfverts[4]->ty );
               surface_copy_properties( s, dls_surf );
               object_add_surface_head( dls_parent, dls_surf );
            }
            break;
         case 4:
            {
               // A quad with 4 new vertices was added
               // Add a new vertex in the center of the quad
               SVertex centerloc;
               centerloc.v = new_vertex_set( 0, 0, 0 );
               centerloc.tx = 0;
               centerloc.ty = 0;
               object_add_vertex( dls_parent, centerloc.v );
               for( int ix = 0; ix < 8; ++ix )
               {
                  centerloc.v->x += newsurfverts[ix]->v->x;
                  centerloc.v->y += newsurfverts[ix]->v->y;
                  centerloc.v->z += newsurfverts[ix]->v->z;
                  centerloc.tx += newsurfverts[ix]->tx;
                  centerloc.ty += newsurfverts[ix]->ty;
               }
               centerloc.v->x /= 8.0f;
               centerloc.v->y /= 8.0f;
               centerloc.v->z /= 8.0f;
               centerloc.tx /= 8.0f;
               centerloc.ty /= 8.0f;
               // Add 4 surfaces
               dls_surf = new_surface();
               list_add_item_head( &new_surfs, dls_surf );
               surface_add_vertex( dls_surf, centerloc.v, centerloc.tx, centerloc.ty );
               surface_add_vertex( dls_surf, newsurfverts[0]->v, newsurfverts[0]->tx, newsurfverts[0]->ty );
               surface_add_vertex( dls_surf, newsurfverts[1]->v, newsurfverts[1]->tx, newsurfverts[1]->ty );
               surface_add_vertex( dls_surf, newsurfverts[2]->v, newsurfverts[2]->tx, newsurfverts[2]->ty );
               surface_copy_properties( s, dls_surf );
               object_add_surface_head( dls_parent, dls_surf );
               dls_surf = new_surface();
               list_add_item_head( &new_surfs, dls_surf );
               surface_add_vertex( dls_surf, centerloc.v, centerloc.tx, centerloc.ty );
               surface_add_vertex( dls_surf, newsurfverts[2]->v, newsurfverts[2]->tx, newsurfverts[2]->ty );
               surface_add_vertex( dls_surf, newsurfverts[3]->v, newsurfverts[3]->tx, newsurfverts[3]->ty );
               surface_add_vertex( dls_surf, newsurfverts[4]->v, newsurfverts[4]->tx, newsurfverts[4]->ty );
               surface_copy_properties( s, dls_surf );
               object_add_surface_head( dls_parent, dls_surf );
               dls_surf = new_surface();
               list_add_item_head( &new_surfs, dls_surf );
               surface_add_vertex( dls_surf, centerloc.v, centerloc.tx, centerloc.ty );
               surface_add_vertex( dls_surf, newsurfverts[4]->v, newsurfverts[4]->tx, newsurfverts[4]->ty );
               surface_add_vertex( dls_surf, newsurfverts[5]->v, newsurfverts[5]->tx, newsurfverts[5]->ty );
               surface_add_vertex( dls_surf, newsurfverts[6]->v, newsurfverts[6]->tx, newsurfverts[6]->ty );
               surface_copy_properties( s, dls_surf );
               object_add_surface_head( dls_parent, dls_surf );
               dls_surf = new_surface();
               list_add_item_head( &new_surfs, dls_surf );
               surface_add_vertex( dls_surf, centerloc.v, centerloc.tx, centerloc.ty );
               surface_add_vertex( dls_surf, newsurfverts[6]->v, newsurfverts[6]->tx, newsurfverts[6]->ty );
               surface_add_vertex( dls_surf, newsurfverts[7]->v, newsurfverts[7]->tx, newsurfverts[7]->ty );
               surface_add_vertex( dls_surf, newsurfverts[0]->v, newsurfverts[0]->tx, newsurfverts[0]->ty );
               surface_copy_properties( s, dls_surf );
               object_add_surface_head( dls_parent, dls_surf );
            }
            break;
         }
         // Remove the old surface
         object_remove_surface( dls_parent, s );
      }
   }

   // Cleanup
   list_free( &surflist );
   ac_edgelist_free( &edge_master );

   // Populate the new selection list
   switch( mSelectMode )
   {
   case SELECT_SURFACE:
      for( List *ps = mSelOld; ps != NULL; ps = ps->next )
      {
         if( s_surflist.mSearch( ps->data ) > -1 )
         {
            list_add_item_head( &mSelNew, ps->data );
         }
      }
      for( List *ps = new_surfs; ps != NULL; ps = ps->next )
      {
         list_add_item_head( &mSelNew, ( Surface * )( ps->data ));
      }
      ac_selection_select_surfacelist( mSelNew );
      break;
   case SELECT_VERTEX:
      mSelNew = list_clone( mSelOld );
      for( List *ps = new_verts; ps != NULL; ps = ps->next )
      {
         list_add_item_head( &mSelNew, ( Vertex * )( ps->data ));
      }
      ac_selection_select_vertexlist( mSelNew );
      break;
   case SELECT_GROUP:
      mSelNew = ac_selection_get_groups();
      break;
   default:
      mSelNew = ac_selection_get_objects();
      break;
   }
   list_free( &new_verts );
   list_free( &new_surfs );

   // Recalc all parent objects' normals
   for( List *po = mObjsNew; po != NULL; po = po->next )
   {
      object_calc_normals_force((( t_mObjectParent * )( po->data ))->o );
   }
   redraw_all();

}
//---------------------------------------------------------------------------------------
void DDHUndoableDivLoop::undo()
{
   mUndoCalled = true;
   // Remove the new objects
   for( List *obs = mObjsNew; obs != NULL; obs = obs->next )
   {
      t_mObjectParent *ob = ( t_mObjectParent * )obs->data;
      object_remove_child_nocleanup( ob->parent, ob->o );
   }
   // Add the old objects
   for( List *obs = mObjsOld; obs != NULL; obs = obs->next )
   {
      t_mObjectParent *ob = ( t_mObjectParent * )obs->data;
      object_add_child( ob->parent, ob->o );
   }
   // Select the old objects/surfaces/verts
   switch( mSelectMode )
   {
   case SELECT_SURFACE:
      ac_selection_select_surfacelist( mSelOld );
      break;
   case SELECT_VERTEX:
      ac_selection_select_vertexlist( mSelOld );
      break;
   default:
      ac_selection_select_objectlist( mSelOld );
      break;
   }
   redraw_all();
}
//---------------------------------------------------------------------------------------
void DDHUndoableDivLoop::redo()
{
   mUndoCalled = false;
   // Remove the old objects
   for( List *obs = mObjsOld; obs != NULL; obs = obs->next )
   {
      t_mObjectParent *ob = ( t_mObjectParent * )obs->data;
      object_remove_child_nocleanup( ob->parent, ob->o );
   }
   // Add the new objects
   for( List *obs = mObjsNew; obs != NULL; obs = obs->next )
   {
      t_mObjectParent *ob = ( t_mObjectParent * )obs->data;
      object_add_child( ob->parent, ob->o );
   }
   // Select the new objects/surfaces/verts
   switch( mSelectMode )
   {
   case SELECT_SURFACE:
      ac_selection_select_surfacelist( mSelNew );
      break;
   case SELECT_VERTEX:
      ac_selection_select_vertexlist( mSelNew );
      break;
   default:
      ac_selection_select_objectlist( mSelNew );
      break;
   }
   redraw_all();
}
//---------------------------------------------------------------------------------------
