#ifdef _WINDOWS
#define WINDOWS
#endif

#include "ac_plugin.h"
#include "Undoables.h"
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#ifdef WINDOWS
#include <windows.h>
#else
#include <time.h>
#endif

static unsigned int ddhrandseed = 0;

//---------------------------------------------------------------------------------------
// Get a random seed number
static unsigned int ddhGetRandomSeed()
{
#ifdef WINDOWS
   return GetTickCount();
#else
   ddhrandseed += time( NULL ) / 2;
   return ddhrandseed;
#endif
}
//---------------------------------------------------------------------------------------
// Append list with object's vertices
static void ddhwave_add_object_recurse( ACObject *o, List **v )
{
   list_append_list_unique( v, ac_object_get_vertexlist( o ));
   List *c = ac_object_get_childrenlist( o );
   for( List *cl = c; cl != NULL; cl = cl->next )
   {
      ACObject *o = ( ACObject * )( cl->data );
      ddhwave_add_object_recurse( o, v );
   }
}
//---------------------------------------------------------------------------------------
// Applies a waveform to selected vertices
static void ddhwave_go( int argc, const char* argv[] )
{

   if( argc < 7 )
   {
      message_dialog( "Insufficient parameters passed to wave utility" );
      return;
   }

   bool random = false;
   unsigned int seed = 0;

   // Get params
   char axis_freq = tolower( argv[1][0] );
   char axis_amp = tolower( argv[2][0] );
   if( axis_freq == axis_amp )
   {
      message_dialog( "wave : cannot apply frequency and amplitude to same axis" );
      return;
   }
   else if( axis_freq != 'x' && axis_freq != 'y' && axis_freq != 'z' )
   {
      message_dialog( "wave : cannot interpret frequency axis %c", axis_freq );
      return;
   }
   else if( axis_amp != 'x' && axis_amp != 'y' && axis_amp != 'z' )
   {
      message_dialog( "wave : cannot interpret amplitude axis %c", axis_amp );
      return;
   }
   float frequency, amplitude, offset;
   bool additive, frequnits;
   if( sscanf( argv[3], "%g", &frequency ) != 1 )
   {
      message_dialog( "wave : unable to parse %s for frequency", argv[3] );
      return;
   }
   if( sscanf( argv[4], "%g", &amplitude ) != 1 )
   {
      message_dialog( "wave : unable to parse %s for amplitude", argv[3] );
      return;
   }
   if( strcmp( argv[5], "random" ) == 0 )
   {
      seed = ddhGetRandomSeed();
      srand( seed );
      offset = (( float )( rand() % 100 )) * 0.01f;
      random = true;
   }
   else
   {
      if( sscanf( argv[5], "%g", &offset ) != 1 )
      {
         message_dialog( "wave : unable to parse %s for offset", argv[3] );
         return;
      }
      offset *= 0.01f;
   }
   additive = !( argv[6][0] == '0' );
   frequnits = !( argv[7][0] == '0' );
   debugf( "Wave:\nFrequency axis = %c\nAmplitude axis = %c\nFrequency = %g\nAmplitude = %g\nOffset = %g\nAdditive = %s\nFreq units = %s\n",
            axis_freq, axis_amp, frequency, amplitude, offset, additive ? "true" : "false", frequnits ? "true" : "false" );

   // Changed vertex list
   List *vertlist = NULL;
   switch( ac_get_select_mode() )
   {
   case SELECT_SURFACE:
      {
         List *l = ac_selection_get_surfaces();
         vertlist = ac_surfacelist_get_vertices( l );
         list_free( &l );
      }
      break;
   case SELECT_VERTEX:
      vertlist = ac_selection_get_vertices();
      break;
   default:
      {
         List *l = ac_selection_get_objects();
         for( List *lp = l; lp != NULL; lp = lp->next )
         {
            ACObject *o = ( ACObject * )( lp->data );
            ddhwave_add_object_recurse( o, &vertlist );
         }
         list_free( &l );
      }
      break;
   }

   if( vertlist == NULL )
   {
      message_dialog( "wave : no vertices selected" );
      return;
   }

   // Add undoable
   add_undoable_vertex_positions( "wave", vertlist );

   float ampbase = 0.0f;
   Vertex *vminaxis = NULL;
   Vertex *vmaxaxis = NULL;
   float ampbasefreq = 0.0f;
   float ampmaxval = 0.0f;

   // Grab the min frequency vertex, use as the base vertex amplitude value
   for( List *lv = vertlist; lv != NULL; lv = lv->next )
   {
      Vertex *v = ( Vertex * )( lv->data );
      if( vminaxis == NULL )
      {
         vminaxis = v;
         vmaxaxis = v;
         switch( axis_freq )
         {
         case 'x':
            ampbasefreq = v->x;
            ampmaxval = v->x;
            break;
         case 'y':
            ampbasefreq = v->y;
            ampmaxval = v->y;
            break;
         case 'z':
            ampbasefreq = v->z;
            ampmaxval = v->z;
            break;
         }
         // Do not continue - otherwise, single vertex runs will screw up
         //continue;
      }
      switch( axis_freq )
      {
      case 'x':
         if( v->x < vminaxis->x )
         {
            vminaxis = v;
            ampbasefreq = v->x;
         }
         if( v->x > vmaxaxis->x )
         {
            vmaxaxis = v;
            ampmaxval = v->x;
         }
         break;
      case 'y':
         if( v->y < vminaxis->y )
         {
            vminaxis = v;
            ampbasefreq = v->y;
         }
         if( v->y > vmaxaxis->y )
         {
            vmaxaxis = v;
            ampmaxval = v->y;
         }
         break;
      case 'z':
         if( v->z < vminaxis->z )
         {
            vminaxis = v;
            ampbasefreq = v->z;
         }
         if( v->z > vmaxaxis->z )
         {
            vmaxaxis = v;
            ampmaxval = v->z;
         }
         break;
      }
   }
   switch( axis_amp )
   {
   case 'x':
      ampbase = vminaxis->x;
      break;
   case 'y':
      ampbase = vminaxis->y;
      break;
   case 'z':
      ampbase = vminaxis->z;
      break;
   }

   // If frequency is not expressed in units, grab the frequency based on the number of waves in the total length
   if( !frequnits )
   {
      offset *= frequency;
      if( ampmaxval - ampbasefreq != 0.0f )
      {
         frequency /= ( ampmaxval - ampbasefreq );
      }
      else
      {
         frequency = 0;
      }
   }

   // Transform all verts
   float f, *a;
   float ampof = 0.0f;
   ACObject *thisobj;
   ACObject *lastobj = NULL;
   bool subdivided = false;
   for( List *lv = vertlist; lv != NULL; lv = lv->next )
   {

      Vertex *v = ( Vertex * )( lv->data );
      thisobj = object_of_vertex( v );

      switch( axis_freq )
      {
      case 'x':
         f = v->x - vminaxis->x;
         break;
      case 'y':
         f = v->y - vminaxis->y;
         break;
      case 'z':
         f = v->z - vminaxis->z;
         break;
      }
      switch( axis_amp )
      {
      case 'x':
         a = &( v->x );
         break;
      case 'y':
         a = &( v->y );
         break;
      case 'z':
         a = &( v->z );
         break;
      default:
         message_dialog( "opos" );
      }

      // Change random offset for each object
      if( thisobj != lastobj )
      {
         subdivided = subdivided | ( ac_object_is_subdivided( thisobj ) != FALSE );
         if( random )
         {
            lastobj = thisobj;
            seed += 7;
            srand( seed );
            offset = (( float )( rand() % 100 )) * 0.01f;
            if( !frequnits )
            {
               offset *= frequency;
            }
         }
      }

      // The wave delta
      float w_sin = sin(((( f * frequency ) + offset ) * 360.0f ) * M_PI / 180.0f ) * amplitude;
      if( additive )
      {
         *a += w_sin;
      }
      else
      {
         *a = ampbase + w_sin;
      }

   }

   // Redraw/refresh
   if( subdivided )
   {
      tcl_command( "ac3d remake_subdivisions" );
   }
   find_bounding_box();
   redraw_all();
   display_status();

}
//---------------------------------------------------------------------------------------
AC3D_PLUGIN_FUNC int AC3DPluginInit( AC3DPluginInitData *d )
{

   static bool firstTime = true;
   if( firstTime )
   {

      char *path = ( char * )strdup( d->plugin_dir_path );
#ifdef WINDOWS
      ac_unbackslash_string( path );
#endif

      tcl_command( "source \"%s/wave.tcl\"", path );
      ac_add_tools_menu_item( "Wave...", "ddhwave_dialog",
                              "Apply an axis-aligned waveform to the selected vertices" );
      ac_add_command_full( "ddhwave_go", ( void * )ddhwave_go,
                            7, "argv", "TODO: Show params here",
                           "Wave utility" );

      firstTime = false;

   }

   return 0;

}
//---------------------------------------------------------------------------------------
AC3D_PLUGIN_FUNC int AC3DPluginExit()
{
   return 0;
}
//---------------------------------------------------------------------------------------
AC3D_PLUGIN_FUNC char *AC3DPluginAbout()
{
   return( "Wave - version 1.1 - Dennis Hawthorne" );
}
//---------------------------------------------------------------------------------------
AC3D_PLUGIN_FUNC char *AC3DPluginInfo()
{
   return( "Wave utility - Applies an axis-aligned waveform to selected vertices" );
}
//---------------------------------------------------------------------------------------

